<?php
/**
 * Enhanced Logging System for BioDynamic Farming Tracker
 * Supports multiple log levels and file rotation
 */

class Logger {
    const LOG_DEBUG = 'DEBUG';
    const LOG_INFO = 'INFO';
    const LOG_WARNING = 'WARNING';
    const LOG_ERROR = 'ERROR';
    const LOG_CRITICAL = 'CRITICAL';

    private static $logDir = __DIR__ . '/../logs/';
    private static $maxFileSize = 10485760; // 10MB
    private static $maxFiles = 10;

    /**
     * Initialize logger (create logs directory if needed)
     */
    public static function init() {
        if (!is_dir(self::$logDir)) {
            mkdir(self::$logDir, 0777, true);
        }
    }

    /**
     * Log debug message
     */
    public static function debug($message, $context = []) {
        self::log(self::LOG_DEBUG, $message, $context);
    }

    /**
     * Log info message
     */
    public static function info($message, $context = []) {
        self::log(self::LOG_INFO, $message, $context);
    }

    /**
     * Log warning message
     */
    public static function warning($message, $context = []) {
        self::log(self::LOG_WARNING, $message, $context);
    }

    /**
     * Log error message
     */
    public static function error($message, $context = []) {
        self::log(self::LOG_ERROR, $message, $context);
    }

    /**
     * Log critical message
     */
    public static function critical($message, $context = []) {
        self::log(self::LOG_CRITICAL, $message, $context);
    }

    /**
     * Log API request
     */
    public static function logRequest($method, $path, $data = null) {
        $context = [
            'method' => $method,
            'path' => $path,
            'ip' => self::getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
        ];
        
        if ($data && !empty($data)) {
            // Don't log sensitive data
            $sanitized = self::sanitizeData($data);
            $context['data'] = $sanitized;
        }

        self::info("API Request: $method $path", $context);
    }

    /**
     * Log API response
     */
    public static function logResponse($statusCode, $path, $responseTime = null) {
        $context = [
            'status_code' => $statusCode,
            'path' => $path,
            'ip' => self::getClientIP()
        ];

        if ($responseTime) {
            $context['response_time_ms'] = $responseTime;
        }

        $level = $statusCode >= 400 ? self::LOG_WARNING : self::LOG_INFO;
        self::log($level, "API Response: $statusCode $path", $context);
    }

    /**
     * Log database query
     */
    public static function logQuery($query, $params = [], $executionTime = null) {
        $context = [
            'query' => $query,
            'params' => $params
        ];

        if ($executionTime) {
            $context['execution_time_ms'] = $executionTime;
        }

        self::debug("Database Query", $context);
    }

    /**
     * Log authentication event
     */
    public static function logAuth($event, $userId = null, $email = null) {
        $context = [
            'event' => $event,
            'ip' => self::getClientIP(),
            'timestamp' => date('Y-m-d H:i:s')
        ];

        if ($userId) {
            $context['user_id'] = $userId;
        }

        if ($email) {
            $context['email'] = $email;
        }

        self::info("Authentication: $event", $context);
    }

    /**
     * Log security event
     */
    public static function logSecurity($event, $severity = 'WARNING', $context = []) {
        $context['event'] = $event;
        $context['ip'] = self::getClientIP();
        $context['timestamp'] = date('Y-m-d H:i:s');

        self::log($severity, "Security Event: $event", $context);
    }

    /**
     * Main logging function
     */
    private static function log($level, $message, $context = []) {
        self::init();

        $timestamp = date('Y-m-d H:i:s');
        $logFile = self::$logDir . strtolower($level) . '.log';

        // Format log message
        $logMessage = self::formatLogMessage($timestamp, $level, $message, $context);

        // Write to log file
        file_put_contents($logFile, $logMessage . PHP_EOL, FILE_APPEND);

        // Also write to main activity log
        $activityLog = self::$logDir . 'activity.log';
        file_put_contents($activityLog, $logMessage . PHP_EOL, FILE_APPEND);

        // Check if file needs rotation
        self::rotateLogFile($logFile);
        self::rotateLogFile($activityLog);

        // Log critical errors to PHP error log as well
        if ($level === self::LOG_CRITICAL || $level === self::LOG_ERROR) {
            error_log($logMessage);
        }
    }

    /**
     * Format log message
     */
    private static function formatLogMessage($timestamp, $level, $message, $context = []) {
        $logMessage = "[$timestamp] [$level] $message";

        if (!empty($context)) {
            $contextStr = json_encode($context, JSON_UNESCAPED_SLASHES);
            $logMessage .= " | Context: $contextStr";
        }

        return $logMessage;
    }

    /**
     * Rotate log file if it exceeds max size
     */
    private static function rotateLogFile($logFile) {
        if (!file_exists($logFile)) {
            return;
        }

        $fileSize = filesize($logFile);

        if ($fileSize > self::$maxFileSize) {
            $baseName = basename($logFile, '.log');
            $timestamp = date('Y-m-d_H-i-s');
            $archivedFile = self::$logDir . $baseName . '_' . $timestamp . '.log';

            rename($logFile, $archivedFile);

            // Clean up old archived files
            self::cleanupOldLogs($baseName);
        }
    }

    /**
     * Clean up old log files
     */
    private static function cleanupOldLogs($baseName) {
        $files = glob(self::$logDir . $baseName . '_*.log');

        if (count($files) > self::$maxFiles) {
            usort($files, function ($a, $b) {
                return filemtime($a) - filemtime($b);
            });

            $filesToDelete = array_slice($files, 0, count($files) - self::$maxFiles);

            foreach ($filesToDelete as $file) {
                unlink($file);
            }
        }
    }

    /**
     * Sanitize sensitive data from logs
     */
    private static function sanitizeData($data) {
        $sensitiveKeys = ['password', 'token', 'secret', 'api_key', 'credit_card', 'ssn'];
        
        if (is_array($data)) {
            $sanitized = [];
            foreach ($data as $key => $value) {
                if (in_array(strtolower($key), $sensitiveKeys)) {
                    $sanitized[$key] = '***REDACTED***';
                } else {
                    $sanitized[$key] = is_array($value) ? self::sanitizeData($value) : $value;
                }
            }
            return $sanitized;
        }

        return $data;
    }

    /**
     * Get client IP address
     */
    private static function getClientIP() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            return $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
        }
    }

    /**
     * Get log file contents
     */
    public static function getLogContents($level = 'activity', $lines = 100) {
        $logFile = self::$logDir . strtolower($level) . '.log';

        if (!file_exists($logFile)) {
            return [];
        }

        $contents = file($logFile);
        return array_slice($contents, -$lines);
    }

    /**
     * Clear log file
     */
    public static function clearLog($level = 'activity') {
        $logFile = self::$logDir . strtolower($level) . '.log';

        if (file_exists($logFile)) {
            file_put_contents($logFile, '');
        }
    }
}

// Initialize logger on include
Logger::init();
?>

