<?php
/**
 * Get Tasks
 * GET /api/tasks/get-tasks.php
 */

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/jwt.php';

header('Content-Type: application/json');
$userId = requireAuth();

try {
    $db = getDBConnection();
    
    // Get query parameters
    $status = isset($_GET['status']) ? $_GET['status'] : null;
    $priority = isset($_GET['priority']) ? $_GET['priority'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    // Build query
    $query = "
        SELECT
            t.id,
            t.title,
            t.description,
            t.category,
            t.status,
            t.priority,
            t.due_date,
            t.reminder_date,
            t.is_recurring,
            t.recurrence_pattern,
            t.created_at,
            t.updated_at,
            c.id as crop_id,
            c.crop_name,
            fs.id as farm_section_id,
            fs.section_name
        FROM tasks t
        LEFT JOIN crops c ON t.crop_id = c.id
        LEFT JOIN farm_sections fs ON t.farm_section_id = fs.id
        WHERE t.user_id = ?
    ";
    
    $params = [$userId];
    
    if ($status) {
        $query .= " AND t.status = ?";
        $params[] = $status;
    }
    
    if ($priority) {
        $query .= " AND t.priority = ?";
        $params[] = $priority;
    }
    
    $query .= " ORDER BY 
        CASE t.priority
            WHEN 'high' THEN 1
            WHEN 'medium' THEN 2
            WHEN 'low' THEN 3
        END,
        t.due_date ASC
        LIMIT ? OFFSET ?
    ";
    
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    foreach ($tasks as &$task) {
        $task['is_recurring'] = (bool)$task['is_recurring'];
        
        // Parse recurrence pattern JSON
        if ($task['recurrence_pattern']) {
            $task['recurrence_pattern'] = json_decode($task['recurrence_pattern'], true);
        }
    }
    
    // Get total count
    $countQuery = "SELECT COUNT(*) as total FROM tasks WHERE user_id = ?";
    $countParams = [$userId];
    
    if ($status) {
        $countQuery .= " AND status = ?";
        $countParams[] = $status;
    }
    
    if ($priority) {
        $countQuery .= " AND priority = ?";
        $countParams[] = $priority;
    }
    
    $stmt = $db->prepare($countQuery);
    $stmt->execute($countParams);
    $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    sendResponse(true, 'Tasks retrieved successfully', [
        'tasks' => $tasks,
        'total' => (int)$total,
        'limit' => $limit,
        'offset' => $offset
    ]);
    
} catch (PDOException $e) {
    error_log('Database error: ' . $e->getMessage());
    sendResponse(false, 'Failed to retrieve tasks');
} catch (Exception $e) {
    error_log('Error: ' . $e->getMessage());
    sendResponse(false, 'An error occurred');
}


