<?php
/**
 * Create Task
 * POST /api/tasks/create-task.php
 */

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/jwt.php';

header('Content-Type: application/json');
$userId = requireAuth();

try {
    $db = getDBConnection();
    
    // Get POST data
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (empty($data['title'])) {
        sendResponse(false, 'Title is required');
    }
    
    if (empty($data['category'])) {
        sendResponse(false, 'Category is required');
    }

    // Validate category
    $validCategories = ['seeding', 'harvesting', 'bd500', 'bd501', 'cpp', 'watering', 'other'];
    if (!in_array($data['category'], $validCategories)) {
        sendResponse(false, 'Invalid category');
    }
    
    // Validate status
    $status = $data['status'] ?? 'pending';
    $validStatuses = ['pending', 'completed', 'postponed', 'cancelled'];
    if (!in_array($status, $validStatuses)) {
        sendResponse(false, 'Invalid status');
    }
    
    // Validate priority
    $priority = $data['priority'] ?? 'medium';
    $validPriorities = ['low', 'medium', 'high'];
    if (!in_array($priority, $validPriorities)) {
        sendResponse(false, 'Invalid priority');
    }
    
    // Validate crop_id if provided
    if (!empty($data['crop_id'])) {
        $stmt = $db->prepare("SELECT id FROM crops WHERE id = ? AND user_id = ?");
        $stmt->execute([$data['crop_id'], $userId]);
        if (!$stmt->fetch()) {
            sendResponse(false, 'Invalid crop ID');
        }
    }
    
    // Validate farm_section_id if provided
    if (!empty($data['farm_section_id'])) {
        $stmt = $db->prepare("SELECT id FROM farm_sections WHERE id = ? AND user_id = ?");
        $stmt->execute([$data['farm_section_id'], $userId]);
        if (!$stmt->fetch()) {
            sendResponse(false, 'Invalid farm section ID');
        }
    }
    
    // Validate due_date if provided
    if (!empty($data['due_date'])) {
        $dueDate = DateTime::createFromFormat('Y-m-d', $data['due_date']);
        if (!$dueDate) {
            sendResponse(false, 'Invalid due date format. Use YYYY-MM-DD');
        }
    }
    
    // Validate reminder_date if provided
    if (!empty($data['reminder_date'])) {
        $reminderDate = DateTime::createFromFormat('Y-m-d H:i:s', $data['reminder_date']);
        if (!$reminderDate) {
            sendResponse(false, 'Invalid reminder date format. Use YYYY-MM-DD HH:MM:SS');
        }
    }
    
    // Handle recurrence
    $isRecurring = isset($data['is_recurring']) ? (bool)$data['is_recurring'] : false;
    $recurrencePattern = null;
    
    if ($isRecurring && !empty($data['recurrence_pattern'])) {
        $recurrencePattern = json_encode($data['recurrence_pattern']);
    }
    
    // Insert task
    $stmt = $db->prepare("
        INSERT INTO tasks (
            user_id, title, description, category, status, priority,
            due_date, reminder_date, is_recurring, recurrence_pattern,
            crop_id, farm_section_id
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");

    $stmt->execute([
        $userId,
        $data['title'],
        $data['description'] ?? null,
        $data['category'],
        $status,
        $priority,
        $data['due_date'] ?? null,
        $data['reminder_date'] ?? null,
        $isRecurring ? 1 : 0,
        $recurrencePattern,
        $data['crop_id'] ?? null,
        $data['farm_section_id'] ?? null
    ]);
    
    $taskId = $db->lastInsertId();
    
    // Get the created task
    $stmt = $db->prepare("
        SELECT
            t.id,
            t.title,
            t.description,
            t.category,
            t.status,
            t.priority,
            t.due_date,
            t.reminder_date,
            t.is_recurring,
            t.recurrence_pattern,
            t.created_at,
            t.updated_at,
            c.id as crop_id,
            c.crop_name,
            fs.id as farm_section_id,
            fs.section_name
        FROM tasks t
        LEFT JOIN crops c ON t.crop_id = c.id
        LEFT JOIN farm_sections fs ON t.farm_section_id = fs.id
        WHERE t.id = ?
    ");
    
    $stmt->execute([$taskId]);
    $task = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    $task['is_recurring'] = (bool)$task['is_recurring'];
    
    // Parse recurrence pattern JSON
    if ($task['recurrence_pattern']) {
        $task['recurrence_pattern'] = json_decode($task['recurrence_pattern'], true);
    }
    
    sendResponse(true, 'Task created successfully', ['task' => $task]);
    
} catch (PDOException $e) {
    error_log('Database error: ' . $e->getMessage());
    sendResponse(false, 'Failed to create task');
} catch (Exception $e) {
    error_log('Error: ' . $e->getMessage());
    sendResponse(false, 'An error occurred');
}


