<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendErrorResponse('Method not allowed', 405);
}

// Get authorization header
$headers = getallheaders();
$authHeader = $headers['Authorization'] ?? '';

if (empty($authHeader) || !preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
    sendErrorResponse('Authorization token required', 401);
}

$token = $matches[1];
$decoded = validateJWT($token);

if (!$decoded) {
    sendErrorResponse('Invalid or expired token', 401);
}

$userId = $decoded['user_id'];

// Get query parameters
$customerId = $_GET['customer_id'] ?? null;
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
$offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;

try {
    // Build query
    $query = "
        SELECT 
            s.id, s.sale_date, s.total_amount, s.payment_method, s.payment_status,
            s.notes, s.created_at, s.updated_at,
            c.id as customer_id, c.customer_name, c.phone, c.email
        FROM sales s
        LEFT JOIN customers c ON s.customer_id = c.id
        WHERE s.user_id = ?
    ";
    
    $params = [$userId];
    
    if ($customerId) {
        $query .= " AND s.customer_id = ?";
        $params[] = $customerId;
    }
    
    $query .= " ORDER BY s.sale_date DESC, s.created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $sales = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get sale items for each sale
    foreach ($sales as &$sale) {
        $stmt = $db->prepare("
            SELECT 
                si.id, si.quantity, si.unit_price, si.subtotal,
                c.id as crop_id, c.crop_name, c.crop_type
            FROM sales_items si
            LEFT JOIN crops c ON si.crop_id = c.id
            WHERE si.sale_id = ?
        ");
        $stmt->execute([$sale['id']]);
        $sale['items'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Get total count
    $countQuery = "SELECT COUNT(*) as total FROM sales WHERE user_id = ?";
    $countParams = [$userId];
    
    if ($customerId) {
        $countQuery .= " AND customer_id = ?";
        $countParams[] = $customerId;
    }
    
    $stmt = $db->prepare($countQuery);
    $stmt->execute($countParams);
    $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    sendSuccessResponse('Sales retrieved successfully', [
        'sales' => $sales,
        'total' => (int)$total,
        'limit' => $limit,
        'offset' => $offset
    ]);
    
} catch (PDOException $e) {
    error_log("Database error: " . $e->getMessage());
    sendErrorResponse('Database error occurred', 500);
}


