<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendErrorResponse('Method not allowed', 405);
}

// Get authorization header
$headers = getallheaders();
$authHeader = $headers['Authorization'] ?? '';

if (empty($authHeader) || !preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
    sendErrorResponse('Authorization token required', 401);
}

$token = $matches[1];
$decoded = validateJWT($token);

if (!$decoded) {
    sendErrorResponse('Invalid or expired token', 401);
}

$userId = $decoded['user_id'];

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    sendErrorResponse('Invalid JSON data', 400);
}

// Validate required fields
$saleDate = $data['sale_date'] ?? null;
$totalAmount = $data['total_amount'] ?? null;
$paymentMethod = $data['payment_method'] ?? null;
$paymentStatus = $data['payment_status'] ?? 'pending';
$items = $data['items'] ?? [];

if (!$saleDate || !$totalAmount || !$paymentMethod || empty($items)) {
    sendErrorResponse('Sale date, total amount, payment method, and items are required', 400);
}

// Validate total amount
if (!is_numeric($totalAmount) || $totalAmount <= 0) {
    sendErrorResponse('Total amount must be a positive number', 400);
}

// Validate items
foreach ($items as $item) {
    if (!isset($item['crop_id']) || !isset($item['quantity']) || !isset($item['unit_price'])) {
        sendErrorResponse('Each item must have crop_id, quantity, and unit_price', 400);
    }
    if (!is_numeric($item['quantity']) || $item['quantity'] <= 0) {
        sendErrorResponse('Item quantity must be a positive number', 400);
    }
    if (!is_numeric($item['unit_price']) || $item['unit_price'] < 0) {
        sendErrorResponse('Item unit price must be a non-negative number', 400);
    }
}

try {
    // Start transaction
    $db->beginTransaction();
    
    // Verify customer if provided
    $customerId = $data['customer_id'] ?? null;
    if ($customerId) {
        $stmt = $db->prepare("SELECT id FROM customers WHERE id = ? AND user_id = ?");
        $stmt->execute([$customerId, $userId]);
        if (!$stmt->fetch()) {
            $db->rollBack();
            sendErrorResponse('Invalid customer ID', 400);
        }
    }
    
    // Insert sale
    $stmt = $db->prepare("
        INSERT INTO sales (
            user_id, customer_id, sale_date, total_amount, payment_method, 
            payment_status, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $userId,
        $customerId,
        $saleDate,
        $totalAmount,
        $paymentMethod,
        $paymentStatus,
        $data['notes'] ?? null
    ]);
    
    $saleId = $db->lastInsertId();
    
    // Insert sale items
    $itemStmt = $db->prepare("
        INSERT INTO sales_items (
            sale_id, crop_id, quantity, unit_price, subtotal
        ) VALUES (?, ?, ?, ?, ?)
    ");
    
    foreach ($items as $item) {
        // Verify crop belongs to user
        $stmt = $db->prepare("SELECT id FROM crops WHERE id = ? AND user_id = ?");
        $stmt->execute([$item['crop_id'], $userId]);
        if (!$stmt->fetch()) {
            $db->rollBack();
            sendErrorResponse('Invalid crop ID: ' . $item['crop_id'], 400);
        }
        
        $subtotal = $item['quantity'] * $item['unit_price'];
        
        $itemStmt->execute([
            $saleId,
            $item['crop_id'],
            $item['quantity'],
            $item['unit_price'],
            $subtotal
        ]);
    }
    
    // Commit transaction
    $db->commit();
    
    // Get created sale with details
    $stmt = $db->prepare("
        SELECT 
            s.id, s.sale_date, s.total_amount, s.payment_method, s.payment_status,
            s.notes, s.created_at, s.updated_at,
            c.id as customer_id, c.customer_name, c.phone, c.email
        FROM sales s
        LEFT JOIN customers c ON s.customer_id = c.id
        WHERE s.id = ?
    ");
    $stmt->execute([$saleId]);
    $sale = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get sale items
    $stmt = $db->prepare("
        SELECT 
            si.id, si.quantity, si.unit_price, si.subtotal,
            c.id as crop_id, c.crop_name, c.crop_type
        FROM sales_items si
        LEFT JOIN crops c ON si.crop_id = c.id
        WHERE si.sale_id = ?
    ");
    $stmt->execute([$saleId]);
    $sale['items'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendSuccessResponse('Sale created successfully', ['sale' => $sale], 201);
    
} catch (PDOException $e) {
    $db->rollBack();
    error_log("Database error: " . $e->getMessage());
    sendErrorResponse('Database error occurred', 500);
}


