<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get query parameters
    if (!isset($_GET['community_id'])) {
        sendErrorResponse('Community ID is required', 400);
    }
    
    $communityId = (int)$_GET['community_id'];
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    // Validate limit
    if ($limit < 1 || $limit > 100) {
        $limit = 20;
    }
    
    // Get database connection
    $db = getDBConnection();
    
    // Check if user is a member of the community
    $stmt = $db->prepare("
        SELECT cm.role 
        FROM community_members cm
        WHERE cm.community_id = ? AND cm.user_id = ? AND cm.status = 'approved'
    ");
    $stmt->execute([$communityId, $userId]);
    $membership = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$membership) {
        sendErrorResponse('You are not a member of this community', 403);
    }
    
    // Get posts
    $stmt = $db->prepare("
        SELECT 
            p.id, p.title, p.content, p.image_url, p.created_at, p.updated_at,
            u.id as user_id, u.name as user_name, u.email as user_email,
            (SELECT COUNT(*) FROM community_post_likes WHERE post_id = p.id) as likes_count,
            (SELECT COUNT(*) FROM community_post_comments WHERE post_id = p.id) as comments_count,
            (SELECT COUNT(*) FROM community_post_likes WHERE post_id = p.id AND user_id = ?) as is_liked
        FROM community_posts p
        INNER JOIN users u ON p.user_id = u.id
        WHERE p.community_id = ?
        ORDER BY p.created_at DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$userId, $communityId, $limit, $offset]);
    $posts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    foreach ($posts as &$post) {
        $post['is_liked'] = (bool)$post['is_liked'];
        $post['likes_count'] = (int)$post['likes_count'];
        $post['comments_count'] = (int)$post['comments_count'];
    }
    
    // Get total count
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM community_posts WHERE community_id = ?");
    $stmt->execute([$communityId]);
    $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    $data = [
        'posts' => $posts,
        'total' => (int)$total,
        'limit' => $limit,
        'offset' => $offset,
    ];
    
    sendSuccessResponse('Posts retrieved successfully', $data);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


