<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (!isset($input['name']) || trim($input['name']) === '') {
        sendErrorResponse('Community name is required', 400);
    }
    
    if (!isset($input['description']) || trim($input['description']) === '') {
        sendErrorResponse('Description is required', 400);
    }
    
    $name = trim($input['name']);
    $description = trim($input['description']);
    $imageUrl = isset($input['image_url']) ? trim($input['image_url']) : null;
    $isPrivate = isset($input['is_private']) ? (bool)$input['is_private'] : true;
    
    // Get database connection
    $db = getDBConnection();
    
    // Start transaction
    $db->beginTransaction();
    
    try {
        // Insert community
        $stmt = $db->prepare("
            INSERT INTO private_communities (name, description, image_url, is_private, created_by)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $name,
            $description,
            $imageUrl,
            $isPrivate ? 1 : 0,
            $userId
        ]);
        
        $communityId = $db->lastInsertId();
        
        // Add creator as admin member
        $stmt = $db->prepare("
            INSERT INTO community_members (community_id, user_id, role, status)
            VALUES (?, ?, 'admin', 'approved')
        ");
        $stmt->execute([$communityId, $userId]);
        
        // Commit transaction
        $db->commit();
        
        // Get the created community
        $stmt = $db->prepare("
            SELECT 
                c.id, c.name, c.description, c.image_url, c.is_private, c.created_by, c.created_at,
                u.name as creator_name,
                (SELECT COUNT(*) FROM community_members WHERE community_id = c.id AND status = 'approved') as members_count,
                (SELECT COUNT(*) FROM community_posts WHERE community_id = c.id) as posts_count
            FROM private_communities c
            INNER JOIN users u ON c.created_by = u.id
            WHERE c.id = ?
        ");
        $stmt->execute([$communityId]);
        $community = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Convert boolean
        $community['is_private'] = (bool)$community['is_private'];
        $community['members_count'] = (int)$community['members_count'];
        $community['posts_count'] = (int)$community['posts_count'];
        $community['user_role'] = 'admin';
        
        sendSuccessResponse('Community created successfully', $community);
        
    } catch (Exception $e) {
        $db->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


