<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (!isset($input['community_id'])) {
        sendErrorResponse('Community ID is required', 400);
    }
    
    if (!isset($input['title']) || trim($input['title']) === '') {
        sendErrorResponse('Title is required', 400);
    }
    
    if (!isset($input['content']) || trim($input['content']) === '') {
        sendErrorResponse('Content is required', 400);
    }
    
    $communityId = (int)$input['community_id'];
    $title = trim($input['title']);
    $content = trim($input['content']);
    $imageUrl = isset($input['image_url']) ? trim($input['image_url']) : null;
    
    // Get database connection
    $db = getDBConnection();
    
    // Check if user is a member of the community
    $stmt = $db->prepare("
        SELECT cm.role 
        FROM community_members cm
        WHERE cm.community_id = ? AND cm.user_id = ? AND cm.status = 'approved'
    ");
    $stmt->execute([$communityId, $userId]);
    $membership = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$membership) {
        sendErrorResponse('You are not a member of this community', 403);
    }
    
    // Insert post
    $stmt = $db->prepare("
        INSERT INTO community_posts (community_id, user_id, title, content, image_url)
        VALUES (?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $communityId,
        $userId,
        $title,
        $content,
        $imageUrl
    ]);
    
    $postId = $db->lastInsertId();
    
    // Get the created post
    $stmt = $db->prepare("
        SELECT 
            p.id, p.title, p.content, p.image_url, p.created_at, p.updated_at,
            u.id as user_id, u.name as user_name, u.email as user_email,
            0 as likes_count,
            0 as comments_count,
            0 as is_liked
        FROM community_posts p
        INNER JOIN users u ON p.user_id = u.id
        WHERE p.id = ?
    ");
    $stmt->execute([$postId]);
    $post = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    $post['is_liked'] = (bool)$post['is_liked'];
    $post['likes_count'] = (int)$post['likes_count'];
    $post['comments_count'] = (int)$post['comments_count'];
    
    sendSuccessResponse('Post created successfully', $post);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


