<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: PUT, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
    sendErrorResponse('Method not allowed', 405);
}

// Get authorization header
$headers = getallheaders();
$authHeader = $headers['Authorization'] ?? '';

if (empty($authHeader) || !preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
    sendErrorResponse('Authorization token required', 401);
}

$token = $matches[1];
$decoded = validateJWT($token);

if (!$decoded) {
    sendErrorResponse('Invalid or expired token', 401);
}

$userId = $decoded['user_id'];

// Get PUT data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    sendErrorResponse('Invalid JSON data', 400);
}

// Validate required fields
$harvestId = $data['harvest_id'] ?? null;

if (!$harvestId) {
    sendErrorResponse('Harvest ID is required', 400);
}

try {
    // Verify harvest belongs to user
    $stmt = $db->prepare("SELECT id FROM harvests WHERE id = ? AND user_id = ?");
    $stmt->execute([$harvestId, $userId]);
    
    if (!$stmt->fetch()) {
        sendErrorResponse('Harvest not found or access denied', 404);
    }
    
    // Build update query dynamically
    $updateFields = [];
    $params = [];
    
    if (isset($data['crop_id'])) {
        // Verify crop belongs to user
        $stmt = $db->prepare("SELECT id FROM crops WHERE id = ? AND user_id = ?");
        $stmt->execute([$data['crop_id'], $userId]);
        if (!$stmt->fetch()) {
            sendErrorResponse('Invalid crop ID', 400);
        }
        $updateFields[] = "crop_id = ?";
        $params[] = $data['crop_id'];
    }
    
    if (isset($data['harvest_date'])) {
        $updateFields[] = "harvest_date = ?";
        $params[] = $data['harvest_date'];
    }
    
    if (isset($data['quantity'])) {
        if (!is_numeric($data['quantity']) || $data['quantity'] <= 0) {
            sendErrorResponse('Quantity must be a positive number', 400);
        }
        $updateFields[] = "quantity = ?";
        $params[] = $data['quantity'];
    }
    
    if (isset($data['unit'])) {
        $updateFields[] = "unit = ?";
        $params[] = $data['unit'];
    }
    
    if (isset($data['quality_rating'])) {
        if (!is_numeric($data['quality_rating']) || $data['quality_rating'] < 1 || $data['quality_rating'] > 5) {
            sendErrorResponse('Quality rating must be between 1 and 5', 400);
        }
        $updateFields[] = "quality_rating = ?";
        $params[] = $data['quality_rating'];
    }
    
    if (isset($data['notes'])) {
        $updateFields[] = "notes = ?";
        $params[] = $data['notes'];
    }
    
    if (empty($updateFields)) {
        sendErrorResponse('No fields to update', 400);
    }
    
    // Add updated_at
    $updateFields[] = "updated_at = NOW()";
    
    // Add harvest ID to params
    $params[] = $harvestId;
    
    // Update harvest
    $sql = "UPDATE harvests SET " . implode(', ', $updateFields) . " WHERE id = ?";
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    
    // Get updated harvest with crop details
    $stmt = $db->prepare("
        SELECT 
            h.id, h.harvest_date, h.quantity, h.unit, h.quality_rating, h.notes,
            h.created_at, h.updated_at,
            c.id as crop_id, c.crop_name, c.crop_type,
            fs.id as farm_section_id, fs.section_name
        FROM harvests h
        LEFT JOIN crops c ON h.crop_id = c.id
        LEFT JOIN farm_sections fs ON c.farm_section_id = fs.id
        WHERE h.id = ?
    ");
    $stmt->execute([$harvestId]);
    $harvest = $stmt->fetch(PDO::FETCH_ASSOC);
    
    sendSuccessResponse('Harvest updated successfully', ['harvest' => $harvest]);
    
} catch (PDOException $e) {
    error_log("Database error: " . $e->getMessage());
    sendErrorResponse('Database error occurred', 500);
}


