<?php
/**
 * Create Harvest
 * POST /api/harvest/create-harvest.php
 */

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/jwt.php';

header('Content-Type: application/json');
$userId = requireAuth();

try {
    $db = getDBConnection();
    
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (empty($data['crop_id'])) {
        sendResponse(false, 'Crop ID is required');
    }
    
    if (empty($data['harvest_date'])) {
        sendResponse(false, 'Harvest date is required');
    }
    
    if (empty($data['quantity'])) {
        sendResponse(false, 'Quantity is required');
    }
    
    if (empty($data['unit'])) {
        sendResponse(false, 'Unit is required');
    }
    
    // Validate crop_id belongs to user
    $stmt = $db->prepare("SELECT id FROM crops WHERE id = ? AND user_id = ?");
    $stmt->execute([$data['crop_id'], $userId]);
    if (!$stmt->fetch()) {
        sendResponse(false, 'Invalid crop ID');
    }
    
    // Validate quality_rating if provided
    if (isset($data['quality_rating'])) {
        $rating = (int)$data['quality_rating'];
        if ($rating < 1 || $rating > 5) {
            sendResponse(false, 'Quality rating must be between 1 and 5');
        }
    }
    
    $stmt = $db->prepare("
        INSERT INTO harvests (
            user_id, crop_id, harvest_date, quantity, unit, quality_rating, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $userId,
        $data['crop_id'],
        $data['harvest_date'],
        $data['quantity'],
        $data['unit'],
        $data['quality_rating'] ?? null,
        $data['notes'] ?? null
    ]);
    
    $harvestId = $db->lastInsertId();
    
    // Get the created harvest with crop details
    $stmt = $db->prepare("
        SELECT 
            h.id, h.harvest_date, h.quantity, h.unit, h.quality_rating, h.notes,
            h.created_at, h.updated_at,
            c.id as crop_id, c.crop_name, c.crop_type,
            fs.id as farm_section_id, fs.section_name
        FROM harvests h
        LEFT JOIN crops c ON h.crop_id = c.id
        LEFT JOIN farm_sections fs ON c.farm_section_id = fs.id
        WHERE h.id = ?
    ");
    
    $stmt->execute([$harvestId]);
    $harvest = $stmt->fetch(PDO::FETCH_ASSOC);
    
    sendResponse(true, 'Harvest created successfully', ['harvest' => $harvest]);
    
} catch (PDOException $e) {
    error_log('Database error: ' . $e->getMessage());
    sendResponse(false, 'Failed to create harvest');
} catch (Exception $e) {
    error_log('Error: ' . $e->getMessage());
    sendResponse(false, 'An error occurred');
}


