<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (!isset($input['receiver_id'])) {
        sendErrorResponse('Receiver ID is required', 400);
    }
    
    if (!isset($input['message']) || trim($input['message']) === '') {
        sendErrorResponse('Message is required', 400);
    }
    
    $receiverId = (int)$input['receiver_id'];
    $message = trim($input['message']);
    $imageUrl = isset($input['image_url']) ? trim($input['image_url']) : null;
    
    // Cannot send message to self
    if ($receiverId === $userId) {
        sendErrorResponse('Cannot send message to yourself', 400);
    }
    
    // Get database connection
    $db = getDBConnection();
    
    // Check if receiver exists
    $stmt = $db->prepare("SELECT id FROM users WHERE id = ?");
    $stmt->execute([$receiverId]);
    if (!$stmt->fetch()) {
        sendErrorResponse('Receiver not found', 404);
    }
    
    // Insert message
    $stmt = $db->prepare("
        INSERT INTO chat_messages (sender_id, receiver_id, message, image_url)
        VALUES (?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $userId,
        $receiverId,
        $message,
        $imageUrl
    ]);
    
    $messageId = $db->lastInsertId();
    
    // Get the created message
    $stmt = $db->prepare("
        SELECT 
            m.id, m.sender_id, m.receiver_id, m.message, m.image_url, m.is_read, m.created_at,
            u.name as sender_name
        FROM chat_messages m
        INNER JOIN users u ON m.sender_id = u.id
        WHERE m.id = ?
    ");
    $stmt->execute([$messageId]);
    $messageData = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Convert boolean
    $messageData['is_read'] = (bool)$messageData['is_read'];
    
    sendSuccessResponse('Message sent successfully', $messageData);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


