<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Method not allowed', null, 405);
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate required fields (support both camelCase and snake_case)
$emailOrPhone = $input['emailOrPhone'] ?? $input['email_or_phone'] ?? '';
$otp = $input['otp'] ?? '';

// Validation
if (empty($emailOrPhone)) {
    sendResponse(false, 'Email or phone number is required');
}

if (empty($otp)) {
    sendResponse(false, 'OTP is required');
}

if (strlen($otp) !== 6 || !ctype_digit($otp)) {
    sendResponse(false, 'Invalid OTP format');
}

try {
    $db = getDBConnection();
    
    // Find OTP verification record
    $stmt = $db->prepare("
        SELECT ov.*, u.id as user_id, u.is_verified
        FROM otp_verification ov
        JOIN users u ON (ov.email = u.email OR ov.phone = u.phone)
        WHERE (ov.email = ? OR ov.phone = ?)
        AND ov.otp = ?
        AND ov.is_used = 0
        AND ov.expires_at > NOW()
        AND u.is_banned = 0
        ORDER BY ov.created_at DESC
        LIMIT 1
    ");
    
    $stmt->execute([$emailOrPhone, $emailOrPhone, $otp]);
    $otpRecord = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$otpRecord) {
        sendResponse(false, 'Invalid or expired OTP');
    }
    
    // Begin transaction
    $db->beginTransaction();
    
    // Mark OTP as used
    $stmt = $db->prepare("UPDATE otp_verification SET is_used = 1 WHERE id = ?");
    $stmt->execute([$otpRecord['id']]);
    
    // Mark user as verified
    $stmt = $db->prepare("UPDATE users SET is_verified = 1, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$otpRecord['user_id']]);
    
    // Commit transaction
    $db->commit();
    
    // Log activity
    logActivity($otpRecord['user_id'], 'otp_verified', 'OTP verified successfully');
    
    // Return success response
    sendResponse(true, 'Verification successful', [
        'user_id' => $otpRecord['user_id']
    ]);
    
} catch (PDOException $e) {
    if (isset($db) && $db->inTransaction()) {
        $db->rollBack();
    }
    error_log("OTP verification error: " . $e->getMessage());
    sendResponse(false, 'Verification failed. Please try again later', null, 500);
}


