<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/jwt.php';

header('Content-Type: application/json');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Method not allowed', null, 405);
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate required fields (support both camelCase and snake_case)
$emailOrPhone = $input['emailOrPhone'] ?? $input['email_or_phone'] ?? '';
$password = $input['password'] ?? '';
$isEmail = $input['isEmail'] ?? $input['is_email'] ?? true;

// Validation
if (empty($emailOrPhone)) {
    sendResponse(false, ($isEmail ? 'Email' : 'Phone number') . ' is required');
}

if (empty($password)) {
    sendResponse(false, 'Password is required');
}

try {
    $db = getDBConnection();
    
    // Find user by email or phone
    if ($isEmail) {
        $stmt = $db->prepare("
            SELECT id, name, email, phone, password_hash, profile_photo, is_verified, is_admin, is_banned, created_at, last_login
            FROM users
            WHERE email = ?
        ");
        $stmt->execute([$emailOrPhone]);
    } else {
        $stmt = $db->prepare("
            SELECT id, name, email, phone, password_hash, profile_photo, is_verified, is_admin, is_banned, created_at, last_login
            FROM users
            WHERE phone = ?
        ");
        $stmt->execute([$emailOrPhone]);
    }
    
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        sendResponse(false, 'Invalid credentials');
    }
    
    // Verify password
    if (!password_verify($password, $user['password_hash'])) {
        sendResponse(false, 'Invalid credentials');
    }
    
    // Check if user is verified
    if (!$user['is_verified']) {
        sendResponse(false, 'Please verify your account first');
    }
    
    // Remove password from user data
    unset($user['password_hash']);

    // Generate JWT tokens
    $accessToken = generateJWT($user['id'], $user['email'] ?? $user['phone'], $user['is_admin']);
    $refreshToken = generateRefreshToken($user['id']);
    
    // Update last login
    $stmt = $db->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
    $stmt->execute([$user['id']]);
    
    // Log activity
    logActivity($user['id'], 'user_login', 'User logged in successfully');
    
    // Convert boolean fields
    $user['is_verified'] = (bool)$user['is_verified'];
    $user['is_admin'] = (bool)$user['is_admin'];
    $user['id'] = (int)$user['id'];
    
    // Return success response with tokens
    sendResponse(true, 'Login successful', [
        'user' => $user,
        'access_token' => $accessToken,
        'refresh_token' => $refreshToken,
        'token_type' => 'Bearer',
        'expires_in' => JWT_EXPIRY_SECONDS
    ]);
    
} catch (PDOException $e) {
    error_log("Login error: " . $e->getMessage());
    sendResponse(false, 'Login failed. Please try again later', null, 500);
}

