<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Method not allowed', null, 405);
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate required fields (support both camelCase and snake_case)
$emailOrPhone = $input['emailOrPhone'] ?? $input['email_or_phone'] ?? '';
$isEmail = $input['isEmail'] ?? $input['is_email'] ?? true;

// Validation
if (empty($emailOrPhone)) {
    sendResponse(false, ($isEmail ? 'Email' : 'Phone number') . ' is required');
}

// Validate email or phone
if ($isEmail) {
    if (!validateEmail($emailOrPhone)) {
        sendResponse(false, 'Invalid email address');
    }
} else {
    if (!validatePhone($emailOrPhone)) {
        sendResponse(false, 'Invalid phone number');
    }
}

try {
    $db = getDBConnection();
    
    // Find user by email or phone
    if ($isEmail) {
        $stmt = $db->prepare("SELECT id, name, email FROM users WHERE email = ?");
        $stmt->execute([$emailOrPhone]);
    } else {
        $stmt = $db->prepare("SELECT id, name, phone FROM users WHERE phone = ?");
        $stmt->execute([$emailOrPhone]);
    }
    
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        // Don't reveal if user exists or not for security
        sendResponse(true, 'If an account exists with this ' . ($isEmail ? 'email' : 'phone number') . ', you will receive a reset code');
    }
    
    // Generate OTP
    $otp = generateOTP();
    $otpExpiry = date('Y-m-d H:i:s', strtotime('+' . OTP_EXPIRY_MINUTES . ' minutes'));
    
    // Insert OTP verification record
    $stmt = $db->prepare("
        INSERT INTO otp_verification (user_id, email, phone, otp, purpose, expires_at, created_at)
        VALUES (?, ?, ?, ?, 'password_reset', ?, NOW())
    ");

    $email = $isEmail ? $emailOrPhone : null;
    $phone = $isEmail ? null : $emailOrPhone;

    $stmt->execute([$user['id'], $email, $phone, $otp, $otpExpiry]);
    
    // Send OTP via email or SMS
    if ($isEmail) {
        // TODO: Send email with OTP
        // For now, we'll just log it
        error_log("Password reset OTP for $emailOrPhone: $otp");
    } else {
        // TODO: Send SMS with OTP
        // For now, we'll just log it
        error_log("Password reset OTP for $emailOrPhone: $otp");
    }
    
    // Log activity
    logActivity($user['id'], 'password_reset_requested', 'Password reset OTP sent');
    
    // Return success response (don't send OTP in production)
    sendResponse(true, 'Reset code sent successfully', [
        'otp' => $otp // Remove this in production
    ]);
    
} catch (PDOException $e) {
    error_log("Forgot password error: " . $e->getMessage());
    sendResponse(false, 'Failed to send reset code. Please try again later', null, 500);
}


