<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get database connection
    $db = getDBConnection();
    
    // Get farm statistics
    $farmStats = getFarmStats($db, $userId);
    
    // Get harvest statistics
    $harvestStats = getHarvestStats($db, $userId);
    
    // Get sales statistics
    $salesStats = getSalesStats($db, $userId);
    
    // Get task statistics
    $taskStats = getTaskStats($db, $userId);
    
    // Get recent activity
    $recentActivity = getRecentActivity($db, $userId);
    
    // Get monthly revenue (last 6 months)
    $monthlyRevenue = getMonthlyRevenue($db, $userId);
    
    // Get crop distribution
    $cropDistribution = getCropDistribution($db, $userId);
    
    $data = [
        'farm_stats' => $farmStats,
        'harvest_stats' => $harvestStats,
        'sales_stats' => $salesStats,
        'task_stats' => $taskStats,
        'recent_activity' => $recentActivity,
        'monthly_revenue' => $monthlyRevenue,
        'crop_distribution' => $cropDistribution,
    ];
    
    sendSuccessResponse('Dashboard data retrieved successfully', $data);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}

function getFarmStats($db, $userId) {
    // Total farm sections
    $stmt = $db->prepare("SELECT COUNT(*) as total_sections FROM farm_sections WHERE user_id = ?");
    $stmt->execute([$userId]);
    $totalSections = $stmt->fetch(PDO::FETCH_ASSOC)['total_sections'];
    
    // Total area
    $stmt = $db->prepare("SELECT SUM(area_size) as total_area FROM farm_sections WHERE user_id = ?");
    $stmt->execute([$userId]);
    $totalArea = $stmt->fetch(PDO::FETCH_ASSOC)['total_area'] ?? 0;
    
    // Total crops
    $stmt = $db->prepare("
        SELECT COUNT(*) as total_crops 
        FROM crops c
        INNER JOIN farm_sections fs ON c.farm_section_id = fs.id
        WHERE fs.user_id = ?
    ");
    $stmt->execute([$userId]);
    $totalCrops = $stmt->fetch(PDO::FETCH_ASSOC)['total_crops'];
    
    // Active crops (not harvested yet)
    $stmt = $db->prepare("
        SELECT COUNT(*) as active_crops 
        FROM crops c
        INNER JOIN farm_sections fs ON c.farm_section_id = fs.id
        WHERE fs.user_id = ? AND c.harvest_date >= CURDATE()
    ");
    $stmt->execute([$userId]);
    $activeCrops = $stmt->fetch(PDO::FETCH_ASSOC)['active_crops'];
    
    return [
        'total_sections' => (int)$totalSections,
        'total_area' => (float)$totalArea,
        'total_crops' => (int)$totalCrops,
        'active_crops' => (int)$activeCrops,
    ];
}

function getHarvestStats($db, $userId) {
    // Total harvests
    $stmt = $db->prepare("SELECT COUNT(*) as total_harvests FROM harvests WHERE user_id = ?");
    $stmt->execute([$userId]);
    $totalHarvests = $stmt->fetch(PDO::FETCH_ASSOC)['total_harvests'];
    
    // Total quantity harvested
    $stmt = $db->prepare("SELECT SUM(quantity) as total_quantity FROM harvests WHERE user_id = ?");
    $stmt->execute([$userId]);
    $totalQuantity = $stmt->fetch(PDO::FETCH_ASSOC)['total_quantity'] ?? 0;
    
    // Average quality rating
    $stmt = $db->prepare("SELECT AVG(quality_rating) as avg_quality FROM harvests WHERE user_id = ? AND quality_rating IS NOT NULL");
    $stmt->execute([$userId]);
    $avgQuality = $stmt->fetch(PDO::FETCH_ASSOC)['avg_quality'] ?? 0;
    
    // This month's harvests
    $stmt = $db->prepare("
        SELECT COUNT(*) as month_harvests 
        FROM harvests 
        WHERE user_id = ? AND MONTH(harvest_date) = MONTH(CURDATE()) AND YEAR(harvest_date) = YEAR(CURDATE())
    ");
    $stmt->execute([$userId]);
    $monthHarvests = $stmt->fetch(PDO::FETCH_ASSOC)['month_harvests'];
    
    return [
        'total_harvests' => (int)$totalHarvests,
        'total_quantity' => (float)$totalQuantity,
        'average_quality' => round((float)$avgQuality, 2),
        'this_month' => (int)$monthHarvests,
    ];
}

function getSalesStats($db, $userId) {
    // Total sales
    $stmt = $db->prepare("SELECT COUNT(*) as total_sales FROM sales WHERE user_id = ?");
    $stmt->execute([$userId]);
    $totalSales = $stmt->fetch(PDO::FETCH_ASSOC)['total_sales'];
    
    // Total revenue
    $stmt = $db->prepare("SELECT SUM(total_amount) as total_revenue FROM sales WHERE user_id = ?");
    $stmt->execute([$userId]);
    $totalRevenue = $stmt->fetch(PDO::FETCH_ASSOC)['total_revenue'] ?? 0;
    
    // This month's revenue
    $stmt = $db->prepare("
        SELECT SUM(total_amount) as month_revenue 
        FROM sales 
        WHERE user_id = ? AND MONTH(sale_date) = MONTH(CURDATE()) AND YEAR(sale_date) = YEAR(CURDATE())
    ");
    $stmt->execute([$userId]);
    $monthRevenue = $stmt->fetch(PDO::FETCH_ASSOC)['month_revenue'] ?? 0;
    
    // Pending payments
    $stmt = $db->prepare("
        SELECT SUM(total_amount) as pending_amount 
        FROM sales 
        WHERE user_id = ? AND payment_status = 'pending'
    ");
    $stmt->execute([$userId]);
    $pendingAmount = $stmt->fetch(PDO::FETCH_ASSOC)['pending_amount'] ?? 0;
    
    return [
        'total_sales' => (int)$totalSales,
        'total_revenue' => (float)$totalRevenue,
        'this_month_revenue' => (float)$monthRevenue,
        'pending_payments' => (float)$pendingAmount,
    ];
}

function getTaskStats($db, $userId) {
    // Total tasks
    $stmt = $db->prepare("SELECT COUNT(*) as total_tasks FROM tasks WHERE user_id = ?");
    $stmt->execute([$userId]);
    $totalTasks = $stmt->fetch(PDO::FETCH_ASSOC)['total_tasks'];
    
    // Pending tasks
    $stmt = $db->prepare("SELECT COUNT(*) as pending_tasks FROM tasks WHERE user_id = ? AND status = 'pending'");
    $stmt->execute([$userId]);
    $pendingTasks = $stmt->fetch(PDO::FETCH_ASSOC)['pending_tasks'];
    
    // Completed tasks
    $stmt = $db->prepare("SELECT COUNT(*) as completed_tasks FROM tasks WHERE user_id = ? AND status = 'completed'");
    $stmt->execute([$userId]);
    $completedTasks = $stmt->fetch(PDO::FETCH_ASSOC)['completed_tasks'];
    
    // Overdue tasks
    $stmt = $db->prepare("
        SELECT COUNT(*) as overdue_tasks 
        FROM tasks 
        WHERE user_id = ? AND status != 'completed' AND due_date < CURDATE()
    ");
    $stmt->execute([$userId]);
    $overdueTasks = $stmt->fetch(PDO::FETCH_ASSOC)['overdue_tasks'];
    
    return [
        'total_tasks' => (int)$totalTasks,
        'pending_tasks' => (int)$pendingTasks,
        'completed_tasks' => (int)$completedTasks,
        'overdue_tasks' => (int)$overdueTasks,
    ];
}

function getRecentActivity($db, $userId) {
    $activities = [];
    
    // Recent harvests (last 5)
    $stmt = $db->prepare("
        SELECT 'harvest' as type, h.harvest_date as date, c.crop_name as description
        FROM harvests h
        LEFT JOIN crops c ON h.crop_id = c.id
        WHERE h.user_id = ?
        ORDER BY h.harvest_date DESC
        LIMIT 5
    ");
    $stmt->execute([$userId]);
    $harvests = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Recent sales (last 5)
    $stmt = $db->prepare("
        SELECT 'sale' as type, s.sale_date as date, CONCAT('Sale: ₹', s.total_amount) as description
        FROM sales s
        WHERE s.user_id = ?
        ORDER BY s.sale_date DESC
        LIMIT 5
    ");
    $stmt->execute([$userId]);
    $sales = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Recent tasks (last 5)
    $stmt = $db->prepare("
        SELECT 'task' as type, t.due_date as date, t.title as description
        FROM tasks t
        WHERE t.user_id = ?
        ORDER BY t.created_at DESC
        LIMIT 5
    ");
    $stmt->execute([$userId]);
    $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Merge and sort by date
    $activities = array_merge($harvests, $sales, $tasks);
    usort($activities, function($a, $b) {
        return strtotime($b['date']) - strtotime($a['date']);
    });
    
    return array_slice($activities, 0, 10);
}

function getMonthlyRevenue($db, $userId) {
    $stmt = $db->prepare("
        SELECT 
            DATE_FORMAT(sale_date, '%Y-%m') as month,
            SUM(total_amount) as revenue
        FROM sales
        WHERE user_id = ? AND sale_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
        GROUP BY DATE_FORMAT(sale_date, '%Y-%m')
        ORDER BY month ASC
    ");
    $stmt->execute([$userId]);
    $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return array_map(function($item) {
        return [
            'month' => $item['month'],
            'revenue' => (float)$item['revenue'],
        ];
    }, $data);
}

function getCropDistribution($db, $userId) {
    $stmt = $db->prepare("
        SELECT 
            c.crop_type,
            COUNT(*) as count
        FROM crops c
        INNER JOIN farm_sections fs ON c.farm_section_id = fs.id
        WHERE fs.user_id = ?
        GROUP BY c.crop_type
    ");
    $stmt->execute([$userId]);
    $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return array_map(function($item) {
        return [
            'type' => $item['crop_type'],
            'count' => (int)$item['count'],
        ];
    }, $data);
}

